using System;
using System.Text;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;
using gov.va.med.vbecs.DAL.VistALink.Client;

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Stanislav Antropov</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>5/11/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// Class incapsulating simple "ping" RPC call to VistA system. 
	/// </summary>
	public class VistALinkPing
	{
		private const string PING_RPC_NAME = "XOBV TEST PING";
		private const string PING_RPC_SECURITY_CONTEXT = "XOBV VISTALINK TESTER";

		private VistALinkPing() {}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4327"> 
		///		<ExpectedInput>Valid responding server connection info</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4328"> 
		///		<ExpectedInput>Valid non-responding server connection info</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4329"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks availability of the VistA system by sending a ping request - 
		/// establishing a connection and executing a ping RPC call provided with 
		/// every installation of VistALink listener. 
		/// 
		/// WARNING: this test method will terminate VistALink connection if it exists.
		/// </summary>
		/// <param name="server">Server to ping.</param>
		/// <param name="errorMessage">Error message - output parameter</param>
		/// <returns>True if ping was successful. Otherwise - false.</returns>
		public static bool Ping( ServerConnectionInfo server, out string errorMessage )
		{			
			if( server == null )
				throw( new ArgumentNullException( "server" ) );

			errorMessage = null;

			// Want to make a safe disconnect, without checks for VistALink initialization.
			VistALink.Disconnect();

			try
			{
				if( !VistALink.UnsafeConnect( server ) )
				{
					// User cancelled logon
					errorMessage = StrRes.SysErrMsg.MUC01.ConnectionTestFailedDueToCancelledLogon().ResString;
					return false;
				}

				VistALink.GetRpcBroker().ExecuteRpc( new RpcRequest( PING_RPC_NAME, PING_RPC_SECURITY_CONTEXT ) );
								
				return true;
			}
			catch( VistALinkException xcp )
			{
				errorMessage = GetErrorMessageFromVistALinkException( xcp );
				return false;
			}
			finally
			{
				VistALink.Disconnect();
			}
		}

		/// <summary>
		/// Extracts user-readable error message from supplied exception. Will walk exception stack if needed. 
		/// Will add exception-specific information if applicable.
		/// </summary>
		/// <param name="xcp">Exception to get error message from.</param>
		/// <returns>Error message extracted from supplied exception.</returns>
		private static string GetErrorMessageFromVistALinkException( Exception xcp ) 
		{
			if( xcp == null )
				throw( new ArgumentNullException( "xcp" ) );

			StringBuilder _sb = new StringBuilder();
			_sb.Append( ( xcp.InnerException == null ? StrRes.SysErrMsg.MUC01.ConnectionTestFailedSingleExceptionHeader() : StrRes.SysErrMsg.MUC01.ConnectionTestFailedMultipleExceptionsHeader() ).ResString );

			Exception _currentException = xcp;

			while( _currentException != null )
			{
				_sb.Append( Environment.NewLine ); _sb.Append( Environment.NewLine );

				if( _currentException is FlatMediumParseException )
				{
					_sb.Append( StrRes.SysErrMsg.MUC01.IncompatibleVistALinkProtocol().ResString );
					_sb.Append( Environment.NewLine ); _sb.Append( Environment.NewLine );
				}

				_sb.Append( _currentException.Message );

				_currentException = _currentException.InnerException;
			}
			
			return _sb.ToString();
		}
	}
}
